## Advanced UI Methods

all components in devflow contains some special methods such as mount/unmount handler and autorun function.

### Mount/Unmount Handler

In devflow, all components have a mount/unmount handler. The mount handler will be called when the component is mounted, and the unmount handler will be called when the component is unmounted.

In following example, you need to modify the ```my_layout``` to trigger reload and unmount/mount.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout, mark_did_mount, mark_will_unmount
class App:
    @mark_create_layout
    def my_layout(self):
        return mui.VBox([
            mui.Typography("Hello APP!"),
        ]).prop(flex=1)

    @mark_did_mount
    async def _on_mount(self):
        print("App mounted")

    @mark_will_unmount
    async def _on_unmount(self):
        print("App unmounted")
```

### Effect (mount-unmount pair)

We can also use mount-unmount pair handler (effect) to handle mount/unmount event. In following example, you need to use `comp.use_effect` and pass a function to it.

The effect handler runs when mount and return a function, which will be called when unmount. if you return `None`, the unmount handler will not be called.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout, mark_did_mount, mark_will_unmount
from functools import partial
import asyncio 
class App:
    @mark_create_layout
    def my_layout(self):
        
        self.layout1 = mui.VBox([
            mui.Typography("Layout1"),
        ]).prop(flex=1)
        self.layout2 = mui.VBox([
            mui.Typography("Layout2"),
        ]).prop(flex=1)
        self.layout1.use_effect(partial(self._effect, "Layout1"))
        self.layout2.use_effect(partial(self._effect, "Layout2"))
        self.container = mui.VBox([
            self.layout1,
        ]).prop(flex=1)
        self.msg = mui.Markdown()
        return mui.VBox([
            mui.Switch("Enable Layout1", self._switch, init_value=True),
            self.msg,
            self.container,
        ])

    async def _switch(self, checked):
        if checked:
            await self.container.set_new_layout([
                self.layout1,
            ])
        else:
            await self.container.set_new_layout([
                self.layout2,
            ])
        

    async def _effect(self, name: str):
        await self.msg.write(self.msg.props.value + f"{name} mounted\n\n")
        async def unmount():
            await self.msg.write(self.msg.props.value + f"{name} unmounted\n\n")

        return unmount
```


### Autorun

In devflow, some special methods are triggered after code change. For example, the layout function will be re-called after code of layout function change. Another special method is ```autorun```, which will be called after code change.

In following example, you need to modify the ```_autorun``` to trigger run of ```_autorun```.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout, mark_autorun
class App:
    @mark_create_layout
    def my_layout(self):
        self.text = mui.Typography("Hello APP!")
        return mui.VBox([
            self.text,
        ]).prop(flex=1)

    @mark_autorun
    async def _autorun_(self):
        await self.text.write("Hello Autorun!!!")
        print("Auto Run!")

```
